/** @file   wire.h
 * @brief   Declaration of Wire - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_WIRE_H
#define H_ENG2D_WIRE_H

#include <vector>
#include "eng2d_dll.h"
#include "interactive.h"
#include "Color.h"

namespace eng2d {

/** @class  Wire
 * @brief   Represents a string or rope with certain number of nodes.
 * @author  Tomi Lamminsaari
 *
 * The wire itself consists of the line segments that connect the nodes
 * to each other.
 */
class DLLIMPORT Wire : public Interactive
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     begPos            The beginning coordinate of this wire.
   * @param     endPos            The ending coordinate of this wire.
   * @param     nodecount         Number of nodes the wire should have.
   */
	Wire( const Vec2D& begPos, const Vec2D& endPos, int nodecount );


	/** Destructor
   */
	virtual ~Wire();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Wire
   */
  Wire( const Wire& rO );

	/** Assignment operator
   * @param     rO                Reference to another Wire
   * @return    Reference to us.
   */
  Wire& operator = ( const Wire& rO );

public:

  ///
  /// Methods
  /// =======

  /** Implements the Interactive - interface. Updates the nodes that construct
   * this wire. This default implementation does nothing.
   */
  virtual void update();
  
  /** Implements the Interactive - interface. Draws this wire on given bitmap.
   * @param     pB                Pointer to target bitmap
   * @param     offset            An offset that is needed when translating
   *                              the coordinates from world-coordinates to
   *                              screen coordinates.
   */
  virtual void redraw( BITMAP* pB, const Vec2D& offset ) const;

  /** Sets the color of this wire.
   * @param     rC                New color.
   */
  virtual void color( const Color& rC );

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the color of this wire.
   * @return    The color
   */
  Color color() const;

  /** Returns the number of nodes this Wire has.
   * @return    How many nodes this Wire has.
   */
  int nodecount() const;

protected:

  /** @struct WireNode
   * @brief   The datastructure used by the individual nodes of the Wire.
   *
   * The wire consists of several individual seqments that are connected
   * to each other. These nodes are instances of this structure.
   */
  struct WireNode {
    /** Position of this node. */
    Vec2D pos;
    /** Possible offset that can be used for any purpose. Usually the actual
     * position of the node is ( pos + offset ).
     */
    Vec2D offset;
  };
  
  /** Constructs given number of nodes between the beginning and
   * ending point. These nodes are pushed to the @c m_nodes - vector.
   * @param     rBeg              Beginning point
   * @param     rEnd              Ending point
   * @param     nodes             Number of nodes
   */
  void createNodes( const Vec2D& rBeg, const Vec2D& rEnd, int nodes );
  
  

  ///
  /// Members
  /// =======

  /** The nodes that form this Wire. The beginning point and ending point
   * are not included in this vector.
   */
  std::vector< WireNode* > m_nodes;
  
  /** The color of this wire */
  Color m_color;
  
  /** Beginning coordinate of this wire. */
  Vec2D m_begPos;
  /** Ending coordinate of this wire */
  Vec2D m_endPos;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
